const { performQuery } = require("../utils/dbFunctions");
const { sendErrorResponse, sendResponse } = require("../utils");
const tables = require("../utils/tables");
const { getSystemTime } = require("../functions/serverTime");
const moment = require("moment-timezone");

module.exports.createDowntimeReason = async (req, res) => {
  try {
    // Get data from request body
    const { 
      category, sub_category, 
      detail_level_1, detail_level_2, 
      detail_level_3, detail_level_4 
    } = req.body;

    // validate data
    if (!category || !sub_category) {
      return sendErrorResponse(res, 400, "Category and Sub-Category are required", 400);
    }

    console.log(detail_level_1);
    

    // Check for duplicate records
    const duplicateRecord = await performQuery(
      `Select * 
      from ${tables.downtime_reasons} 
      Where 
        category = ? and sub_category = ? AND 
        detail_level_1 = ? AND detail_level_2 = ? AND 
        detail_level_3 = ? AND detail_level_4 = ? AND is_deleted = 0`,
      [category, sub_category, detail_level_1, detail_level_2, detail_level_3, detail_level_4]
    );
    if (duplicateRecord.length > 0) {
      return sendErrorResponse(res, 409, "Record Already Exists", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert data into database
    const result = await performQuery(
      `INSERT INTO ${tables.downtime_reasons} SET ? `,
      {
        category: category,
        sub_category: sub_category,
        detail_level_1: detail_level_1,
        detail_level_2: detail_level_2,
        detail_level_3: detail_level_3,
        detail_level_4: detail_level_4,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime,
      }
    );

    // send Success Response
    return sendResponse(res, { id: result.insertId }, "Reason Created Successfully", 201);
    
  } catch (error) {
    console.error("Error in Create Reason API:", error);
    return sendErrorResponse(res, error, "Error in Create Reason API");
  }
};

module.exports.getDownTimeReasons = async (req, res) => {
  try {
    const query = `
      SELECT 
        dr.id, dr.category, dr.sub_category, 
        dr.detail_level_1, dr.detail_level_2, 
        dr.detail_level_3, dr.detail_level_4,
        dr.is_active,
        dr.updated_by, u.name AS Last_updated_by,
        dr.updated_at AS Last_updated_at 
      FROM ${tables.downtime_reasons} dr
      LEFT JOIN ${tables.users} u ON dr.updated_by = u.id 
      WHERE dr.is_deleted = 0 ORDER BY dr.updated_at DESC`;
    const data = await performQuery(query);
    if (data.length === 0) {
      return sendResponse(res, data, "No Data Found", 200);
    }
    return sendResponse(res, data, "selected Successfully", 200);
  } catch (error) {
    console.error("Error getting reason:", error);
    return sendErrorResponse(res, error, "Error getting reason");
  }
};

module.exports.updateDownTimeReasons = async (req, res) => {
  try {
    // console.log("req.body updateReasonTree >>>", req.body);
    const { 
      id, category, sub_category, 
      detail_level_1, detail_level_2, 
      detail_level_3, detail_level_4 
    } = req.body;

    // validate data
    if (!id || !category || !sub_category) {
      return sendErrorResponse(res, 400, "ID, Category and Sub-Category are required", 400);
    }

    // Check for duplicate records
    const duplicateRecord = await performQuery(
      `Select * 
      from ${tables.downtime_reasons} 
      Where 
        category = ? and sub_category = ? AND 
        detail_level_1 = ? AND detail_level_2 = ? AND 
        detail_level_3 = ? AND detail_level_4 = ? AND id <> ? AND is_deleted = 0`,
      [category, sub_category, detail_level_1, detail_level_2, detail_level_3, detail_level_4, id]
    );
    if (duplicateRecord.length > 0) {
      return sendErrorResponse(res, 409, "Record Already Exists", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update data in database
    const data = await performQuery(
      `UPDATE ${tables.downtime_reasons} SET ? WHERE id = ?`,
      [{
        category:  category,
        sub_category: sub_category,
        detail_level_1: detail_level_1,
        detail_level_2: detail_level_2,
        detail_level_3: detail_level_3,
        detail_level_4: detail_level_4,
        updated_by: req?.user?.id,
        updated_at: currentTime,
      }, id]
    );

    // send Success Response
    return sendResponse(res, data, "Updated Successfully", 200);

  } catch (error) {
    console.error("Error updating reason:", error);
    return sendErrorResponse(res, error, "Error updating reason");
  }
};

module.exports.deleteDownTimeReasons = async (req, res) => {
  try {
    const { id } = req.query;

    // validate data
    if (!id) {
      return sendErrorResponse(res, 400, "Reason ID is required", 400);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const data = await performQuery(
      `UPDATE ${tables.downtime_reasons} 
        SET is_deleted = 1, updated_by = ?, updated_at = ? 
        WHERE id = ?`,
      [req?.user?.id, currentTime, id]
    );

    if (data.affectedRows > 0) {
      return sendResponse(res, data, "Deleted Successfully", 200);
    }

    return sendErrorResponse(res, 400, "Error in Deleting Reason", 400);
  } catch (error) {
    console.error("Error deleting reason:", error);
    return sendErrorResponse(res, error, "Error deleting reason");
  }
};
