
const moment = require("moment-timezone");

const safeNumber = (value) => {
  const num = Number(value);
  return isNaN(num) ? 0 : num;
};

const formatDate = (date) => {
  return date ? moment(date).format("DD-MMM-YYYY") : null;
};

const calculateOEE = (data, orderData) => {
  const runningEvents = data.filter((r) => r.event_name === "Running");
  const stopEvents = data.filter(
    (r) => r.event_name !== "Running" && r.stop_category !== "Utilization"
  );
  const nptEvents = data.filter(
    (r) => r.event_name !== "Running" && r.stop_category === "Utilization"
  );
  const productionEvents = data.filter(
    (r) => r.stop_category !== "Utilization" && r.main_stop_id == null
  );

  const scrapArray = orderData?.map((x) => safeNumber(x.scrap)) || [];
  const uptimeArray = runningEvents.map((x) => safeNumber(x.duration));
  const downtimeArray = stopEvents.map((x) => safeNumber(x.downtime));
  const nptArray = nptEvents.map((x) => safeNumber(x.downtime));
  const productionArray = productionEvents.map((x) =>
    safeNumber(x.total_production)
  ); 
  const weightedSpeedArray = runningEvents.map((x) =>
    isNaN(parseFloat(x.product_speed))
      ? 0
      : parseFloat(x.product_speed) * (safeNumber(x.duration) / 60)
  );

  const upTime = uptimeArray.reduce((a, b) => a + b, 0) / 60;
  const downTime = downtimeArray.reduce((a, b) => a + b, 0) / 60;
  const nptTime = nptArray.reduce((a, b) => a + b, 0) / 60;
  const plannedProductionTime = upTime + downTime;
  const totalProduction = productionArray.reduce((a, b) => a + b, 0);
  const totalScrap = scrapArray.reduce((a, b) => a + b, 0);
  const weightedSpeed = weightedSpeedArray.reduce((a, b) => a + b, 0);

  const avgProductSpeed =
    weightedSpeed === 0 || upTime === 0 ? 0 : weightedSpeed / upTime;
  const avgProductionSpeed =
    totalProduction === 0 || upTime === 0 ? 0 : totalProduction / upTime;

  const availability = parseFloat(
    plannedProductionTime === 0
      ? 0
      : (upTime / plannedProductionTime) * 100);
  
  const performance = parseFloat(
    avgProductSpeed === 0
      ? 0
      : (avgProductionSpeed / avgProductSpeed) * 100);
  
  const quality = parseFloat(
    totalProduction === 0
      ? 0
      : ((totalProduction - totalScrap) / totalProduction) * 100);  
  
  const oee = parseFloat((availability * performance * quality) / 10000);

  const cappedQuality = Math.max(-200, quality);

  return {
    uptime: upTime,
    downtime: downTime,
    nptTime: nptTime,
    plannedProductionTime: plannedProductionTime,
    totalProduction: totalProduction,
    avgProductionSpeed: avgProductionSpeed,
    avgProductSpeed: avgProductSpeed,
    availability: availability,
    performance: performance,
    quality: cappedQuality,
    oee: oee,
  };
};

const groupBy = (data, key) => {
  return data.reduce((result, item) => {
    const keyValue = item[key];
    if (!result[keyValue]) {
      result[keyValue] = [];
    }
    result[keyValue].push(item);
    return result;
  }, {});
};

module.exports =  { safeNumber, formatDate, calculateOEE, groupBy };
