const moment = require("moment"); // Import required libraries
const { getSystemTime, getTimezone } = require("../functions/serverTime");
const tables = require("../utils/tables");
const { performQuery } = require("../utils/dbFunctions");

const getShiftData = async (timestamp) => {
  try {
    
    const currentTime = moment(timestamp).format("HH:mm:ss");
    const getShiftQuery = `
      SELECT * 
      FROM ${tables.shift}
      WHERE
        is_active = 1 AND  
        (
          (
            -- For shifts that do NOT span midnight
            TIME(shift_start) <= TIME(?) 
            AND TIME(shift_end) > TIME(?)
            AND shift_end > shift_start
          ) 
          OR (
            -- For shifts that span midnight
            TIME(shift_start) > TIME(shift_end)
            AND (
                TIME(?) >= TIME(shift_start) 
                OR TIME(?) < TIME(shift_end)
            )
          )
        )
    `;

    // console.log("getShiftData Query ==> : ", getShiftQuery);
    // console.log("getShiftData Parameters ==> : ", [currentTime, currentTime, currentTime, currentTime]);

    const shiftData = await performQuery(getShiftQuery, [currentTime, currentTime, currentTime, currentTime]);
    // console.log("getShiftData Result ==> : ", shiftData);

    return shiftData;
  } catch (error) {
    console.error("Error in getShiftData: ", error);
    throw error;
  }
};

const getShiftTimes = async (shift, timestamp) => {
  // const currentDate = moment(await getSystemTime()).add(40,"minutes"); // Get today's date using moment
  const currentDate = moment(timestamp); // Get today's date using moment
  const timezone = await getTimezone(); // Replace with your desired timezone

  // Parse the start and end times
  let startDateTime = moment.tz(
    `${currentDate.format("YYYY-MM-DD")} ${shift.shift_start}`,
    timezone
  );
  let endDateTime = moment.tz(
    `${currentDate.format("YYYY-MM-DD")} ${shift.shift_end}`,
    timezone
  );
  

  // If the end time is earlier than the start time, add 1 day to the endDateTime
  if (endDateTime.isSameOrBefore(startDateTime)) {
    if(currentDate < endDateTime){
      startDateTime.subtract(1, "day");
    }
    else{
      endDateTime.add(1, "day");
    }
  }

  

  // Format the dates into "MM/DD/YYYY HH:mm:ss"
  const formattedStart = startDateTime.format("YYYY-MM-DD HH:mm:ss");
  const formattedEnd = endDateTime.format("YYYY-MM-DD HH:mm:ss");

  return {
    id: shift.id,
    shift_name: shift.shift_name,
    shift_start: formattedStart,
    shift_end: formattedEnd,
  };
};

const getCurrentShift = async (shiftData) => {
  try {
    const systemTimestamp = await getSystemTime();
    const timezone = await getTimezone(); // Replace with your desired timezone
    const currentTime = moment(systemTimestamp).format("HH:mm:ss");
    const getShiftQuery = `
      SELECT * 
      FROM ${tables.shift}
      WHERE 
      is_active = 1 AND 
      (
        (
            -- For shifts that do NOT span midnight
            TIME(shift_start) <= TIME(?) 
            AND TIME(shift_end) > TIME(?)
            AND shift_end > shift_start
        ) OR (
            -- For shifts that span midnight
            TIME(shift_start) > TIME(shift_end)
            AND (
                TIME(?) >= TIME(shift_start) 
                OR TIME(?) < TIME(shift_end)
            )
        )
      )
    `;
    
    
    const currentShiftData = await performQuery(getShiftQuery, [currentTime, currentTime, currentTime, currentTime]);

    console.log("currentShiftData: ", currentShiftData);
    

    const result = await shiftData.map(async (shift) => {
      if(shift.id == currentShiftData?.[0]?.id){
        shift.currentShift = 1
      }
      else{
        shift.currentShift = 0
      }
      
      let startDateTime = moment.tz(
          `${systemTimestamp.format("YYYY-MM-DD")} ${shift.shift_start}`,
          timezone
        );
      let endDateTime = moment.tz(
        `${systemTimestamp.format("YYYY-MM-DD")} ${shift.shift_end}`,
        timezone
      );
      
      if (endDateTime.isAfter(startDateTime)) {
        if(systemTimestamp < startDateTime){
          startDateTime.subtract(1, "day");
          endDateTime.subtract(1, "day");
        }
      }
      
      // If the end time is earlier than the start time, add 1 day to the endDateTime
      if (endDateTime.isSameOrBefore(startDateTime)) {
        if(systemTimestamp < endDateTime){
          startDateTime.subtract(1, "day");
        }
        else{
          endDateTime.add(1, "day");
        }
      }
      
      // Format the dates into "MM/DD/YYYY HH:mm:ss"
      shift.shift_start = startDateTime.format("YYYY-MM-DD HH:mm:ss");
      shift.shift_end = endDateTime.format("YYYY-MM-DD HH:mm:ss");
    });

    return Promise.all(result).then(() => { return shiftData; });

  } catch (error) {
    console.error("Error in getShiftData: ", error);
    throw error;
  }
};


module.exports = {
  getShiftTimes, getShiftData
};




